/*
 *  upload.js
 *
 *  Copyright: (c) 2018 FileMaker, Inc. All rights reserved.
 *
 *  FILEMAKER CONFIDENTIAL
 *  This file includes unpublished proprietary source code of FileMaker,
 *  Inc. The copyright notice above does not evidence any actual or
 *  intended publication of such source code.
 */
'use strict';
var errorHandler = require('../routes/error');
var util = require('../routes/util');
var fs = require('fs');
var path = require('path');

var deleteUploads = function(fullPath, uploadDir, files) {
	for (var file in files) {
		var f = files[file];
		if (fs.existsSync(f.path)) {
			fs.unlinkSync(f.path);
	    }
  	};
};

/**
 * @api {POST} /:version/databases/:solution/layouts/:layout/records/:recordId/containers/:containerFieldName/:repetition upload to container field
 * @apiDescription Upload body contents into container field :containerFieldName with field repetition :repetition
 * @apiName Upload
 * @apiGroup Upload
 *
 * @apiUse AccessTokenHeader
 *
 * @apiParam {application/form-data} payload containing file to upload into container <br>
 *
 * @apiSuccess {String} errorCode "0"
 * @apiSuccess {String} data JSON Array of row objects
 *
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *       "result": "OK",
 *       "data": [record1, record2, ...]
 *     }
 *
 * @apiUse BadRequestError
 * @apiUse UnauthorizedError
 * @apiUse FMServiceError
 *
 *
 * @apiSampleRequest /:version/databases/:solution/layouts/:layout/records/:recordId/containers/:containerFieldName/:repetition
 */
module.exports.uploadToContainerField = function(req, res, next){
	var ipStr = util.parseClientIp(req);
	var params = {
		'solution': req.swagger.params.database.value,
		'layout': req.swagger.params.layout.value,
		'requestMethod': req.method,
		'requestPath': req.originalUrl,
		'requestIp': ipStr,
		'recordId' : req.swagger.params.recordId.value,
		'containerField': req.swagger.params.containerFieldName.value,
		'repetition': (req.swagger.params.containerFieldRepetition) ? req.swagger.params.containerFieldRepetition.value : '1'
	};

	var result = util.setAuthentication(req, res, params);	
	if(result){

		if (req.swagger.params.modId && req.swagger.params.modId.value) {
			params.modId = req.swagger.params.modId.value;
		}
		if (isNaN(params.recordId) || params.recordId <= 0) {
			return errorHandler.handleError('BadRequest', req, res, 'recordId must be an integer with value greater than 0.',"960")			
		}
		if (isNaN(params.repetition) || params.repetition <= 0) {
			return errorHandler.handleError('BadRequest', req, res, 'repetition must be an integer with value greater than 0.',"960")			
		}

		try{
			util.thrift_client.uploadPreflight(params,
				function(thrifError, thrifResult){
				try{
					var result = JSON.parse(thrifResult);
					if (result.messages[0].code !== '0') {
						return util.handleThrifReturn(thrifError, thrifResult, req, res);
					} else {
						if (req.swagger.params.upload !== undefined) {
							var file = req.swagger.params.upload.value
							if (!/[^\u0000-\u00ff]/.test(file.originalname)) {
								file.originalname = Buffer.from(file.originalname, 'latin1').toString('utf8');
							}
							var fullPath = result.uploadDir.endsWith(path.sep) ? result.uploadDir + file.originalname : form.uploadDir + path.sep + file.originalname;
							if (result.maxFieldLength > 0 && file.buffer.size > result.maxFieldLength) {
								errorHandler.handleError('BadRequest', req, res, "Upload file size larger than field max length of '"+ result.maxFieldLength +"' bytes.", "511");
							} else {
								fs.writeFileSync(fullPath, file.buffer);
								params['fullPath'] = fullPath;
								util.thrift_client.insertIntoContainerField(params,
									function(thrifError, thrifResult) {
										return util.handleThrifReturn(thrifError, thrifResult, req, res);
									}
								);
							}
						} else {
							return errorHandler.handleError('BadRequest', req, res, "Upload payload must contain a part named 'upload'.", "960");								
						}
					}
				} catch(e) {
					errorHandler.handleError('InternalServerError', req, res, "Invalid json string returned from internal server.", "-1");
				}
				});
		} catch (err){
			util.thriftExceptionHandler(err, req, res);
		}
	}
};

module.exports.uploadToContainerFieldRep = module.exports.uploadToContainerField;
